/*
TIA Library is Copyright (c) 2015 Billy D. Spelchan

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

// This is a third-party class simple class library that I am using for this 
// library.
// -----------------------------------------------------------------------------
/* Simple JavaScript Inheritance
 * By John Resig http://ejohn.org/
 * MIT Licensed.
 */
// Inspired by base2 and Prototype
(function(){
  var initializing = false, fnTest = /xyz/.test(function(){xyz;}) ? /\b_super\b/ : /.*/;
 
  // The base Class implementation (does nothing)
  this.Class = function(){};
 
  // Create a new Class that inherits from this class
  Class.extend = function(prop) {
    var _super = this.prototype;
   
    // Instantiate a base class (but only create the instance,
    // don't run the init constructor)
    initializing = true;
    var prototype = new this();
    initializing = false;
   
    // Copy the properties over onto the new prototype
    for (var name in prop) {
      // Check if we're overwriting an existing function
      prototype[name] = typeof prop[name] == "function" &&
        typeof _super[name] == "function" && fnTest.test(prop[name]) ?
        (function(name, fn){
          return function() {
            var tmp = this._super;
           
            // Add a new ._super() method that is the same method
            // but on the super-class
            this._super = _super[name];
           
            // The method only need to be bound temporarily, so we
            // remove it when we're done executing
            var ret = fn.apply(this, arguments);        
            this._super = tmp;
           
            return ret;
          };
        })(name, prop[name]) :
        prop[name];
    }
   
    // The dummy class constructor
    function Class() {
      // All construction is actually done in the init method
      if ( !initializing && this.init )
        this.init.apply(this, arguments);
    }
   
    // Populate our constructed prototype object
    Class.prototype = prototype;
   
    // Enforce the constructor to be what we expect
    Class.prototype.constructor = Class;
 
    // And make this class extendable
    Class.extend = arguments.callee;
   
    return Class;
  };
})();

// -----------------------------------------------------------------------------
// Now here is MY code. - first lets create the scan-line class
// -----------------------------------------------------------------------------

// set up "package" for the library
if (typeof(TIALib) == "undefined") TIALib = {};

// constant indicating register should be kept unchanged
TIALib.KEEP = -256;

/*
The color values used are based on List of video game console palettes
http://en.wikipedia.org/wiki/List_of_video_game_console_palettes
*/
TIALib.COLORS = [
// Hue 0
"#000", "#000", "#404040", "#404040","#6c6c6c", "#6c6c6c","#909090", "#909090",
"#b0b0b0","#b0b0b0","#c8c8c8","#c8c8c8","#dcdcdc","#dcdcdc","#ececec","#ececec",
// Hue 1
"#440","#440","#646410","#646410","#848424","#848424","#a0a034","#a0a034",
"#b8b840","#b8b840","#d0d050","#d0d050","#e8e85c","#e8e85c","#fcfc68","#fcfc68",
//Hue 2
"#702800","#702800","#844414","#844414","#985c28","#985c28","#ac783c","#ac783c",
"#bc8c4c","#bc8c4c","#cca05c","#cca05c","#dcb468","#dcb468","#ecc878","#ecc878",
// Hue 3
"#841800","#841800","#983418","#983418","#ac5030","#ac5030","#c06848","#c06848",
"#d0805c","#d0805c","#e09470","#e09470","#eca880","#eca880","#fcbc94","#fcbc94",
// Hue 4
"#880000","#880000","#9c2020","#9c2020","#b03c3c","#b03c3c","#c05858","#c05858",
"#d07070","#d07070","#e08888","#e08888","#eca0a0","#eca0a0","#fcb4b4","#fcb4b4",
// Hue 5
"#78005c","#78005c","#8c2074","#8c2074","#a03c88","#a03c88","#b0589c","#b0589c",
"#c070b0","#c070b0","#d084c0","#d084c0","#dc9cd0","#dc9cd0","#ecb0e0","#ecb0e0",
// Hue 6
"#480078","#480078","#602090","#602090","#783ca4","#783ca4","#8c58b8","#8c58b8",
"#a070cc","#a070cc","#b484dc","#b484dc","#c49cec","#c49cec","#d4b0fc","#d4b0fc",
// Hue 7
"#140084","#140084","#302098","#302098","#4c3cac","#4c3cac","#6858c0","#6858c0",
"#7c70d0","#7c70d0","#9488e0","#9488e0","#a8a0ec","#a8a0ec","#bcb4fc","#bcb4fc",
// Hue 8
"#000088","#000088","#1c209c","#1c209c","#3840b0","#3840b0","#505cc0","#505cc0",
"#6874d0","#6874d0","#7c8ce0","#7c8ce0","#90a4ec","#90a4ec","#a4b8fc","#a4b8fc",
// Hue 9
"#00187c","#00187c","#1c3890","#1c3890","#3854a8","#3854a8","#5070bc","#5070bc",
"#6888cc","#6888cc","#7c9cdc","#7c9cdc","#90b4ec","#90b4ec","#a4c8fc","#a4c8fc",
// Hue 10
"#002c5c","#002c5c","#1c4c78","#1c4c78","#386890","#386890","#5084ac","#5084ac",
"#689cc0","#689cc0","#7cb4d4","#7cb4d4","#90cce8","#90cce8","#a4e0fc","#a4e0fc",
// Hue 11
"#003c2c","#003c2c","#1c5c48","#1c5c48","#387c64","#387c64","#509c80","#509c80",
"#68b494","#68b494","#7cd0ac","#7cd0ac","#90e4c0","#90e4c0","#a4fcd4","#a4fcd4",
// Hue 12
"#003c00","#003c00","#205c20","#205c20","#407c40","#407c40","#5c9c5c","#5c9c5c",
"#74b474","#74b474","#8cd08c","#8cd08c","#a4e4a4","#a4e4a4","#b8fcb8","#b8fcb8",
// Hue 13
"#143800","#143800","#345c1c","#345c1c","#507c38","#507c38","#6c9850","#6c9850",
"#84b468","#84b468","#9ccc7c","#9ccc7c","#b4e490","#b4e490","#c8fca4","#c8fca4",
// Hue 14
"#2c3000","#2c3000","#4c501c","#4c501c","#687034","#687034","#848c4c","#848c4c",
"#9ca864","#9ca864","#b4c078","#b4c078","#ccd488","#ccd488","#e0ec9c","#e0ec9c",
// Hue 15
"#442800","#442800","#644818","#644818","#846830","#846830","#a08444","#a08444",
"#b89c58","#b89c58","#d0b46c","#d0b46c","#e8cc7c","#e8cc7c","#fce08c","#fce08c"
];

TIALib.ScanLine = Class.extend( {

// ***** VARIABLES *****
	// TIA color registers
	_colorP0 : "#F0F",
	_colorP1 : "#F0F",
	_colorPF : "#F0F",
	_colorBK : "#808",

	// playfield bits (stored as left and right side in proper bit order
	_pfLeft : 0,
	_pfRight : 0,
	_pfMirror :  0,
	
	// sprite player 0
	_player0Scale : 1,
	_player0Position : 0,
	_player0Bits : 0,
	_player0Mirror : 0,
	_player0VDelay : false,
	_player0Hmove : 0,
	
	// sprite player 1
	_player1Scale : 1,
	_player1Position : 0,
	_player1Bits : 0,
	_player1Mirror : 0,
	_player1VDelay : false,
	_player1Hmove : 0,
	
	// missile 0
	_missile0Size : 1, 
	_missile0Enabled : false, 
	_missile0Position : 0,
	_missile0Hmove : 0,

	// missile 1
	_missile1Size : 1, 
	_missile1Enabled : false, 
	_missile1Position : 0,
	_missile1Hmove : 0,
	
	// ball
	_ballSize : 1, 
	_ballEnabled : false, 
	_ballPosition : 0,
	_ballVDelay : false,
	_ballHmove : 0,
	
// ***** CONSTRUCTOR *****

// ***** METHODS - TIA Utility functions *****
	//clearScanLine() - resets scan-line to empty state.
	/** makes the scan-line hold the same state as the passed scan-line except
	 * the vertical delays are all set to false (as they should have passed)
	 * @param scanline = scanline to copy 
	*/
	copyScanLine : function(scanline) {
		// copy register values where approrpiate (all vertical delays are set
		// to false as it is assumed at least one scan line has passed.
		this._colorP0 = scanline._colorP0;
		this._colorP1 = scanline._colorP1;
		this._colorPF = scanline._colorPF;
		this._colorBK = scanline._colorBK;
		this._pfLeft = scanline._pfLeft;
		this._pfRight = scanline._pfRight;
		this._pfMirror = scanline._pfMirror;
		this._player0Scale = scanline._player0Scale;
		this._player0Position = scanline._player0Position;
		this._player0Bits = scanline._player0Bits;
		this._player0Mirror = scanline._player0Mirror;
		this._player0VDelay = false,
		this._player0Hmove = scanline._player0Hmove;
		this._player1Scale = scanline._player1Scale;
		this._player1Position = scanline._player1Position;
		this._player1Bits = scanline._player1Bits;
		this._player1Mirror = scanline._player1Mirror;
		this._player1VDelay = false;
		this._player1Hmove = scanline._player1Hmove;
		this._missile0Size = scanline._missile0Size;
		this._missile0Enabled = scanline._missile0Enabled;
		this._missile0Position = scanline._missile0Position;
		this._missile0Hmove = scanline._missile0Hmove;
		this._missile1Size = scanline._missile1Size;
		this._missile1Enabled = scanline._missile1Enabled;
		this._missile1Position = scanline._missile1Position;
		this._missile1Hmove = scanline._missile1Hmove;
		this._ballSize = scanline._ballSize;
		this._ballEnabled = scanline._ballEnabled;
		this._ballPosition = scanline._ballPosition;
		this._ballVDelay = false;
		this._ballHmove = scanline._ballHmove;
	},

	/** Simple utility function for making sure value within certain range.
	 * @param value = value being checked
	 * @param low = lowest value allowed
	 * @param high = highest value allowed
	 * @return clampped value
	 */
	clamp : function(value, low, high) {
		if (value < low)
			return low;
		if (value > high)
			return high;
		return value;
	},
	
	/** flips (reverses order) of the number of bits indicated.
	 * @param value Value to have bits flipped
	 * @param bits number of bits to be flipped
	 * @return flipped value
	 */
	flipBits : function(value, bits) {
		var bit;
		var shiftValue = value;
		var result = 0;
		
		// loop through bits
		for (var cntr = 0; cntr < bits; ++cntr) {
			// bit is lowest bit in shifting value
			bit = shiftValue & 1;
			// shift value right 
			shiftValue >>= 1;
			// shift result left then add bit to result 
			result = (result << 1) | bit;
		}
		
		return result;
	},
	
// ***** METHODS - TIA Register setters *****
	
	/** Sets scan-line colours (use TIALib.KEEP to use existing register value)
	 * @param player0 color of player 0 sprite and missile
	 * @param player1 color of player 1 sprite and missile
	 * @param playfield color of playerfield and ball sprite
	 * @param background color of background
	*/
	setColors : function(player0, player1, playfield, background) {
		if (player0 >= 0)
			this._colorP0 = TIALib.COLORS[this.clamp(player0,0,255)];
		if (player1 >= 0)
			this._colorP1 = TIALib.COLORS[this.clamp(player1,0,255)];
		if (playfield >= 0)
			this._colorPF = TIALib.COLORS[this.clamp(playfield,0,255)];
		if (background >= 0)
			this._colorBK = TIALib.COLORS[this.clamp(background,0,255)];
	},

	/** Sets playfield using three registers with their mixed orders
	 * (use negative value or TIALib.KEEP to use existing register value)
	 * @param pf0 byte holding PF0 (upper 4 bits in reverse order)
	 * @param pf1 byte holding PF1 (proper order)
	 * @param pf2 byte holding PF2 (reversed order)
	 * @param isMirror <0 = keep, 0 = normal, 1 = mirrored
	*/
	setPlayfieldManually : function(pf0, pf1, pf2, isMirror) {
		// pf0 needs to be reversed then shifted to make room for pf1
		var temp;
		if (pf0 >= 0)
			temp = this.flipBits(pf0, 8) << 8;
		else
			temp = (this._pfLeft >> 8) & 0xff00
		
		// pf1 is in right order so just ORed then shifted for pf2 room
		if (pf1 >= 0)
			temp = (temp | pf1) << 8;
		else
			temp = (temp | ((this._pfLeft >> 8) & 0xff)) << 8

		// pf2 needs to be flipped
		if (pf2 >= 0)
			temp |= this.flipBits(pf2, 8);
		else
			temp |= (this._pfLeft & 0xff);
		
		// we are done our bit work for the left
		this._pfLeft = temp;
		
		// if mirrored, flip temp otherwise right is temp.
		var mirror = isMirror;
		if (isMirror < 0)
			mirrored = this._pfMirror;
		if (mirror == 1) {
			this._pfRight = this.flipBits(temp, 20);
		} else {
			this._pfRight = temp;
		}
	},
	
	/**  All twenty bits in proper (bit 19..bit0) order 
	 * @param bits the 20 bits that make up the playfield data (<0 = keep)
	 * @param isMirror (<0=keep, 0=normal, 1=mirrored)
	*/
	setPlayfieldQuickly : function(bits, isMirror) {
		// left bits are just provided bits (neg value means keep existing)
		if (bits >= 0)
			this._pfLeft = bits;

		// adjust mirroring based on if kept 
		var mirror = isMirror;
		if (isMirror < 0)
			mirror = this._pfMirror;
		this._pfMirror = mirror;
		
		// right bits may need to be flipped based on mirror setting
		if (mirror) {
			this._pfRight = this.flipBits(this._pfLeft, 20);
		} else {
			this._pfRight = this._pfLeft;
		}
	},
	
	/** Sets playfield using three registers with their mixed orders
	 * (negative values indicate keep existing value for that register
	 * @param pf0l PF0 for left side of screen
	 * @param pf1l PF1 for left side of screen
	 * @param pf2l PF2 for left side of screen
	 * @param pf0r PF0 for right side of screen
	 * @param pf1r PF1 for right side of screen
	 * @param pf2r PF2 for right side of screen
	*/
	setAsyncPlayfieldManually : function(pf0l, pf1l, pf2l, pf0r, pf1r, pf2r) {
		// first set right side using other playfield command
		this.setPlayfieldManually(pf0r, pf1r, pf2r, false);
		// reserve right side value
		var temp = this._pfRight;
		// set the left bits
		this.setPlayfieldManually(pf0l, pf1l, pf2l, false);
		// overwrite rigth bits with proper value
		this._pfRight = temp;
	},
	
	/** All twenty bits in proper (bit 19..bit0) order for both sides 
	 * (negative value indicates keep existing values)
	 * @param bitsLeft playfield bits for left side of the screen
	 * @param bitsRight playfield bits for right side of the screen
	*/
	setAsyncPlayfieldQuickly : function(bitsLeft, bitsRight) {
		// left bits and right bits are just provided bits
		if (bitsLeft >= 0)
			this._pfLeft = bitsLeft;
		if (bitsRight >= 0)
			this._pfRight = bitsRight;
	},

	/** Set sprite 0 using clock/offset
	 * @param size 1, 2, or 4 
	 * @param reflect should bits be mirrored
	 * @param bits image bits for the sprite (only 8 bits used)
	 * @param ticks number of CPU tick (1 to 76)
	 * @param hmo the hmove adjustment (-8 to 7) 
	*/
	setPlayer0SpriteManual : function(size, reflect, bits, ticks, hmo, vdel) {
		// adjust hmove value
		this._player0Hmove = this.clamp(hmo, -8, 7);
		var pos = this.clamp(ticks, 1, 76) * 3 - 68 + this._player0Hmove;
		// now just call quick version of this method to do the work
		this.setPlayer0SpriteQuick(size, reflect, bits, pos);
		this._player0VDelay = vdel;
	},
	
	/** Set sprite 0 using x coordinate
	 * @param size 1, 2, or 4 
	 * @param reflect should bits be mirrored
	 * @param bits image bits for the sprite (only lower 8 bits used) 
	 * @param position x coordinate of sprite, -68 to 160 
	*/
	setPlayer0SpriteQuick : function( size, reflect, bits, position) {
		this._player0Scale = this.clamp(size, 1, 4);
		if (this._player0Scale == 3)
			this._player0Scale = 2;
		
		this._player0Position = position;
		if (reflect) {
			this._player0Bits = this.flipBits(bits, 8);
		} else {
			this._player0Bits = bits;
		}
		this._player0VDelay = false;
	},
	
	/** Sets player 0 sprite data
	 * @param bits image bits for the sprite (only lower 8 bits used) 
	 */
	setPlayer0SpriteData : function(bits) {
		this._player0Bits = bits;
	},
	
	/** Set sprite 1 using clock/offset
	 * @param size 1, 2, or 4 
	 * @param reflect should bits be mirrored
	 * @param bits image bits for the sprite (only 8 bits used)
	 * @param ticks number of CPU tick (1 to 76)
	 * @param hmo the hmove adjustment (-8 to 7) 
	*/
	setPlayer1SpriteManual : function(size, reflect, bits, ticks, hmo, vdel) {
		// adjust hmove value
		this._player1Hmove = this.clamp(hmo, -8, 7);
		var pos = this.clamp(ticks, 1, 76) * 3 - 68 + this._player1Hmove;
		// now just call quick version of this method to do the work
		this.setPlayer1SpriteQuick(size, reflect, bits, pos);
		this._player0VDelay = vdel;
	},
	
	/** Set sprite 1 using x coordinate
	 * @param size 1, 2, or 4 
	 * @param reflect should bits be mirrored
	 * @param bits image bits for the sprite (only lower 8 bits used) 
	 * @param position x coordinate of sprite, -68 to 160 
	*/
	setPlayer1SpriteQuick : function( size, reflect, bits, position) {
		this._player1Scale = this.clamp(size, 1, 4);
		if (this._player1Scale == 3)
			this._player1Scale = 2;
		
		this._player1Position = position;
		if (reflect) {
			this._player1Bits = this.flipBits(bits, 8);
		} else {
			this._player1Bits = bits;
		}
		this._player1VDelay = false;
	},

	/** Sets player 1 sprite data
	 * @param bits image bits for the sprite (only lower 8 bits used) 
	 */
	setPlayer1SpriteData : function(bits) {
		this._player1Bits = bits;
	},
	
	/* Set missile 0 using clock/offset */
	setMissile0Manual : function(size, enabled, tick, hmo) {
		this.setMissile0Quick(size, enabled,  ticks * 3 - 68 + hmo);
	},

	setPlayer0StriteData : function(bits) {
		this._player0Bits = bits;
	},
	
	/* Set missile 0 using x coordinate */
	setMissile0Quick : function( size, enabled, position) {
		this._missile0Size = size; // todo verify size 
		this._missile0Enabled = enabled; 
		this._missile0Position = position;
	},
	
	/* Set missile 1 using clock/offset */
	setMissile1Manual : function(size, enabled, tick, hmo) {
		this.setMissile1Quick(size, enabled,  ticks * 3 - 68 + hmo);
	},
	
	/* Set missile 1 using x coordinate */
	setMissile1Quick : function( size, enabled, position) {
		this._missile1Size = size; // todo verify size 
		this._missile1Enabled = enabled; 
		this._missile1Position = position;
	},
	
	enableMissiles : function(m0enabled, m1enabled) {
		this._missile0Enabled = m0enabled;
		this._missile1Enabled = m1enabled;
	},
	
	/* Set ball using clock/offset */
	setBallManual : function(size, enabled, tick, hmo, vdel) {
		this.setBallQuick(size, enabled, tick*3-68+hmo);
	},
	
	/* Set ball using x coordinate */
	setBallQuick : function( size, enabled, position) {
		this._ballSize = size; // todo verify size 
		this._ballEnabled = enabled; 
		this._ballPosition = position;
		this._ballVDelay = false;
	},

// ***** METHODS - Rendering scanline *****

	/*  Renders scanline */
	renderToCanvas : function(ctx, x, y, scaleX, scaleY) {
		//ctx.fillRect(x,y, 160*scaleX,scaleY);
		var cntr;
		var left = x;
		var pfBit = 1 << 19;
		// draw left side playfield bits
		for (cntr = 0; cntr < 20; ++cntr) {
			if ((this._pfLeft & pfBit) > 0) { 
				ctx.fillStyle=this._colorPF;
			} else {
				ctx.fillStyle=this._colorBK;
			}
			ctx.fillRect(left,y, 4*scaleX,scaleY);
			left += (4*scaleX);
			pfBit >>= 1;
		}
		// draw right side playfield bits
		pfBit = 1 << 19;
		for (cntr = 0; cntr < 20; ++cntr) {
			if ((this._pfRight & pfBit) > 0) { 
				ctx.fillStyle=this._colorPF;
			} else {
				ctx.fillStyle=this._colorBK;
			}
			ctx.fillRect(left,y, 4*scaleX,scaleY);
			left += (4*scaleX);
			pfBit >>= 1;
		}

		// draw ball sprite
		if ((this._ballEnabled) && (this._ballVDelay == false)) {
			ctx.fillStyle = this._colorPF;
			ctx.fillRect(x + this._ballPosition * scaleX,y, 
					this._ballSize*scaleX,scaleY);
		}
		
		// draw player sprite 0		
		var spriteBit = 128;
		ctx.fillStyle = this._colorP0;
		if (this._player0VDelay == false) {
			left = this._player0Position * scaleX;
			for (cntr = 0; cntr < 8; ++cntr) {
				if ((this._player0Bits & spriteBit) > 0) {
					ctx.fillRect(left,y, this._player0Scale*scaleX,scaleY);
				}
				left += (this._player0Scale*scaleX);
				spriteBit >>= 1;
			}
		}
		
		// draw missile sprite 0
		if (this._missile0Enabled) {
			ctx.fillStyle = this._colorP0;
			ctx.fillRect(x + this._missile0Position * scaleX,y, 
					this._missile0Size*scaleX,scaleY);
		}

		// draw player sprite 1
		var spriteBit = 128;
		ctx.fillStyle = this._colorP1;
		if (this._player1VDelay == false) {
			left = this._player1Position * scaleX;
			for (cntr = 0; cntr < 8; ++cntr) {
				if ((this._player1Bits & spriteBit) > 0) {
					ctx.fillRect(left,y, this._player1Scale*scaleX,scaleY);
				}
				left += (this._player1Scale*scaleX);
				spriteBit >>= 1;
			}
		}
		
		// draw missile sprite 0
		if (this._missile1Enabled) {
			ctx.fillStyle = this._colorP1;
			ctx.fillRect(x + this._missile1Position * scaleX,y, 
					this._missile1Size*scaleX,scaleY);
		}
	
	}
} );


